#
# IGRA v2.2 Weather Balloon Analysis
#
# Data: ftp.ncei.noaa.gov/pub/data/irga
#
#
# This program reads the monthly results datasets built by make_mo_files.r
@ and placed in month_results
# It then maps the results
#
# IGRA v2.2 Weather Balloon Monthly Maps (1990–2025)
# Improved & Optimized Version - December 2025
#
# © Copyright 2025 Andy May

install.packages("viridis")

library(data.table)
library(sf)
library(tidyterra)
library(ggplot2)
library(terra)
library(rnaturalearth)
library(rnaturalearthdata)
library(viridis)
library(dplyr)

root_dir <- "D:/Climate_Change/weather_balloons/NOAA/Analysis/"
input_dir <- file.path(root_dir, "month_results")
output_dir <- file.path(input_dir, "maps")

if (!dir.exists(output_dir)) dir.create(output_dir, recursive = TRUE)

# Robinson projection (defined once)
rob_crs <- "+proj=robin +lon_0=0 +x_0=0 +y_0=0 +datum=WGS84 +units=m +no_defs"

# Load world map and transform once
world <- ne_countries(scale = "medium", returnclass = "sf")
world_rob <- st_transform(world, rob_crs)

# Graticule (lat: -60 to 60, lon: every 30°)
graticule <- st_graticule(lat = seq(-60, 60, by = 15),
                          lon = seq(-180, 180, by = 30),
                          crs = st_crs(4326)) %>%
  st_transform(rob_crs)

# Load ITCZ latitudes and legend scales
itcz_lat <- fread(file.path(root_dir, "ITCZ_mean_latitudes.csv"))
legend_limits <- fread(file.path(root_dir, "variable_lists_details/map_color_scale_top_base.csv"))

# Define which variables to map (by row index in legend_limits)
# Update this list if you add/remove variables
map_indices <- c(2, 3, 6, 7, 8, 15, 16, 17, 25, 26, 27, 35, 36, 37, 41, 42, 43, 44)

maps_to_make <- legend_limits[map_indices]
if (nrow(maps_to_make) != length(map_indices)) {
  stop("Some selected map indices are out of range in legend_limits.")
}

# Pre-compute template raster for interpolation (100 km resolution)
ext_rob <- ext(vect(world_rob))
template_rast <- rast(ext_rob, resolution = 100000, crs = rob_crs)

month_idx <- 1
# Main loop over months
for (month_idx in 1:12) {
  
  file_name <- file.path(input_dir, paste0("Month_", month_idx, "_map_values.rds"))
  
  if (!file.exists(file_name)) {
    message("Skipping missing file: ", file_name)
    next
  }
  
  message("Processing ", month.name[month_idx], " ...")
  
  dt <- readRDS(file_name)
  
  if (nrow(dt) == 0) {
    message("No data for ", month.name[month_idx])
    next
  }
  
  # Aggregate by station: mean of all computed variables
    agg <- dt[, lapply(.SD, mean, na.rm = TRUE), by = station_id]

  # Keep only relevant columns: location + all int_* and mean_* variables
    cols_to_keep <- names(agg)[grepl("^(latitude|longitude|d_month|int_|mean_)", names(agg))]
    agg <- agg[, ..cols_to_keep]

  # Vector-averaged wind speed and direction (upper, middle, lower troposphere)
    wind_levels <- c("upper", "middle", "lower")
    for (lvl in wind_levels) {
      u_col <- paste0("mean_u_", lvl)
      v_col <- paste0("mean_v_", lvl)
    
      agg[[paste0("spd_", lvl)]] <- sqrt(agg[[u_col]]^2 + agg[[v_col]]^2)
    
    # Direction FROM (meteorological: 0° = from north, 90° = from east)
      agg[[paste0("dir_", lvl)]] <- (atan2(agg[[u_col]], agg[[v_col]]) * 180 / pi + 180 + 360) %% 360
    }
    setdiff(maps_to_make$Name, names(agg))  # should return "n_obs" (or similar)
    intersect(maps_to_make$Name, names(agg))  # should show the good ones
  # Convert to sf (WGS84 → Robinson)
  points_sf <- st_as_sf(agg, coords = c("longitude", "latitude"), crs = 4326)
  points_rob <- st_transform(points_sf, rob_crs)
  points_vect <- vect(points_rob)
  
  # Create ITCZ line for this month
  itcz_this_month <- itcz_lat$ITCZ_lat[month_idx]
  itcz_line <- data.frame(
    lon = seq(-180, 180, by = 1),
    lat = rep(itcz_this_month, 361)
  ) %>%
    st_as_sf(coords = c("lon", "lat"), crs = 4326) %>%
    summarise(geometry = st_combine(geometry)) %>%
    st_cast("LINESTRING") %>%
    st_transform(rob_crs)
  i<-1
  # Loop over variables to map
  for (i in seq_len(nrow(maps_to_make))) {
    
    var_name <- maps_to_make$Name[i]
    var_title <- maps_to_make$Title[i]
    var_min <- maps_to_make$base[i]
    var_max <- maps_to_make$top[i]
    
    if (!var_name %in% names(agg)) {
      message("Variable ", var_name, " not found in data. Skipping.")
      next
    }
    
    # Interpolate using Inverse Distance Weighting
    interpolated <- interpIDW(template_rast, points_vect, var_name,
                              radius = 300000, power = 2)
    names(interpolated) <- var_name
    
    # Define color breaks
    breaks <- seq(var_min, var_max, length.out = 11)
    
    # Build plot
    p <- ggplot() +
      geom_spatraster(data = interpolated, aes(fill = .data[[var_name]]), alpha = 0.8) +
      geom_sf(data = world_rob, fill = "gray50", color = "gray90", linewidth = 0.5) +
      geom_spatraster_contour(data = interpolated,
                              aes(z = .data[[var_name]]),
                              breaks = breaks,
                              color = "black",
                              size = 0.3) +
      geom_sf(data = points_rob,
              aes(color = .data[[var_name]]),
              size = 1.8,
              show.legend = "point") +
      geom_sf(data = itcz_line, color = "white", linewidth = 1.5, linetype = "solid") +
      geom_sf(data = itcz_line, color = "black", linewidth = 0.8, linetype = "solid") +  # outline for visibility
      geom_sf(data = graticule, color = "gray60", linewidth = 0.2, alpha = 0.6) +
      geom_sf(
          data = sf::st_graticule(lat = seq(-60, 60, by = 15), 
               lon = seq(-180, 180, by = 30), crs = rob),
               color = "black",
               linewidth = 0.2,
               alpha = 0.5
      ) +
      scale_fill_viridis_c(option = "plasma",
                           name = var_title,
                           limits = c(var_min, var_max),
                           guide = guide_colorbar(title.position = "top", title.hjust = 0.5)) +
      scale_color_viridis_c(option = "plasma",
                            name = var_title,
                            limits = c(var_min, var_max),
                            guide = "none") +
      coord_sf(crs = rob_crs, expand = FALSE) +
      labs(title = paste(var_title, "–", month.name[month_idx], "1990–2025"),
           subtitle = paste("Mean ITCZ Latitude:", round(itcz_this_month, 1), "°N")) +
      theme_void(base_size = 13) +
      theme(
        plot.background = element_rect(fill = "lightblue", color = NA),
        panel.background = element_rect(fill = "lightblue", color = NA),
        plot.title = element_text(size = 16, face = "bold", hjust = 0.5, color = "navy"),
        plot.subtitle = element_text(size = 13, hjust = 0.5, color = "navy"),
        legend.position = "bottom",
        legend.title = element_text(size = 12, color = "navy"),
        legend.text = element_text(size = 10, color = "navy"),
        legend.key.width = unit(1.5, "cm")
      )
      # print(p)
    
    # Save
    out_file <- file.path(output_dir,
                          paste0(month.name[month_idx], "_", var_name, ".png"))
    
    ggsave(out_file, plot = p, width = 12, height = 7, dpi = 300, bg = "white")
    
    message("  → Saved: ", basename(out_file))
  }
  
  # Cleanup
  rm(dt, agg, points_sf, points_rob, points_vect)
  gc()
}

message("All monthly maps generated successfully!")